************************************************************************
**                                                                    **
**                  Chesapeake Bay Sediment Model                     **
**                                                                    **
** Third version (speeded up) received from Fitzpatrick May 8, 1990   **
**              First modified by C. Cerco May 9, 1990                **
**           Final modifications D.M. Di Toro, Jan 27, 1992           **
**                                                                    **
** Deposit feeders removed by CFC 11/05/14                            **
**                                                                    **
** Modified (updated methanogenesis-sulfate reduction algorithms)     **
**      -- HydroQual, Inc. (N. Churi, D. Di Toro) Latest: 18 Apr 97   **
**                                                                    **
** Error in methane saturation discovered by J. Martin fixed          **
** according to remedy provided by J. Fitzpatrick.  June 5, 2003      **
**                                                                    **
** Sediment PIP added by CFC May 3, 2006                              **
**                                                                    **
** org flux to sediment ==> diagenesis ==> inorg flux to water column **
**                                                                    **
**     3-G model - G1=labile, G2=refractory, G3=slow refractory       **
**                                                                    **
** Silica removed by CFC 11/17/14                                     **
**                                                                    **
** This code has changes to denitrification and P sorption            **
** as per Testa et al 2013.  Completed by CFC 071216                  **
**                                                                    **************************************************************************
**                                                                    **
** Inputs                                                             **
**                                                                    **
**            Required inputs for sediment sub-model                  **
**                                                                    **
** A.  Passed to sediment subroutine from water quality subroutine    **
**                                                                    **
**   1.  Overlying water column segment volume (V1)                   **
**   2.  Overlying water column segment depth (BL(ISEG,3))            **
**   3.  Overlying water column depth                                 **
**   4.  Overlying water column segment temperature and salinity      **
**   5.  Overlying water column ammonia, nitrate, phosphate,          **
**       and dissolved oxygen concentrations                          **
**                                                                    **
** B. Inputs supplied via direct input to the sediment subroutine     **
**                                                                    **
**  Variable names        Description                         Units   **
**                                                                    **
**                                                                    **
**     HSEDALL      Depth of sediment layer  (h2)               cm    **
**      DIFFT       Water column-sediment layer diffusion     cm2/sec **
**                  coefficient for temperature                       **
**     SALTSW       Salinity concentration for determining      ppt   **
**                  whether methane or sulfide SOD formulation        **
**                  is to be used.                                    **
**     SALTND       Determines whether fresh or saltwater             **
**                  nitrification/denitrification rates are used      **
**     SALTLL       Lower limit for salinity with lower P sorption    **
**     SALTUL       Upper limit for salinity with lower P sorption    **
**                                                                    **
**   Diagenesis stoichiometry                                         **
**                                                                    **
**                 Fractions of G1, G2, and G3 contained in ...       **
**                                                                    **
**   FRPPH1(3)     Algal group no.1 phosphorus                        **
**   FRPPH2(3)     Algal group no.2 phosphorus                        **
**   FRPPH3(3)     Algal group no.3 phosphorus                        **
**   FRNPH1(3)     Algal group no.1 nitrogen                          **
**   FRNPH2(3)     Algal group no.2 nitrogen                          **
**   FRNPH3(3)     Algal group no.3 nitrogen                          **
**   FRCPH1(3)     Algal group no.1 carbon                            **
**   FRCPH2(3)     Algal group no.2 carbon                            **
**   FRCPH3(3)     Algal group no.3 carbon                            **
**                                                                    **
**   Diagenesis kinetics                                              **
**                                                                    **
**   KPDIAG(3)     Reaction rates for POP G1, G2, and G3        /day  **
**   DPTHTA(3)     Temperature thetas for POP G1, G2, and G3          **
**   KNDIAG(3)     Reaction rates for PON G1, G2, and G3        /day  **
**   DNTHTA(3)     Temperature thetas for PON G1, G2, and G3          **
**   KCDIAG(3)     Reaction rates for POC G1, G2, and G3        /day  **
**   DCTHTA(3)     Temperature thetas for POC G1, G2, and G3          **
**                                                                    **
**   Solids and transport                                             **
**                                                                    **
**   VPMIX(NSED)   Particulate diffusion coefficient (Dp)   m**2/day  **
**   THTADP        Temperature theta for Dp                           **
**   VDMIX(NSED)   Porewater diffusion coefficient (Dd)     m**2/day  **
**   THTADD        Temperature theta for Dd                           **
**      M1         Concentration of solids in layer 1       kg/l      **
**      M2         Concentration of solids in layer 2       kg/l      **
**                                                                    **
**   Reaction kinetics                                                **
**                                                                    **
**    KAPPNH4F     Nitrification reaction velocity                    **
**                 for freshwater in layer 1                m/day     **
**    KAPPNH4S     Nitrification reaction velocity                    **
**                 for saltwater  in layer 1                m/day     **
**    PIENH4       Ammonia partition coefficient            L/kg      **
**    THTANH4      Theta for nitrification reaction velicities        **
**    KMNH4        Nitrification half saturation constant             **
**                 for ammonia                              mg N/m3   **
**    KMNH4O2      Nitrification half saturation constant             **
**                 for oxygen                               mg O2/L   **
**    K1NO3        Denitrification reaction velocity                  **
**                 in layer 1                               m/day     **
**    K2NO3        Denitrification reaction velocity                  **
**                 in layer 2                               m/day     **
**    THTANO3      Theta for denitrification                          **
**    KAPPD1       Reaction velocity for dissolved sulfide            **
**                 oxidation in layer 1                     m/day     **
**    KAPPP1       Reaction velocity for particulate sulfide          **
**                 oxidation in layer 1                     m/day     **
**    PIE1S        Partition coefficient for sulfide                  **
**                 in layer 1                               L/kg      **
**    PIE2S        Partition coefficient for sulfide                  **
**                 in layer 2                               L/kg      **
**    THTAPD1      Theta for both dissolved and particulate           **
**                 sulfide oxidation                                  **
**    KMHSO2       Sulfide oxidation normalization constant           **
**                 for oxygen                               mg O2/L   **
**    DPIE1PO4     Incremental partition coefficient for              **
**                 phosphate in layer 1                     L/kg      **
**    PIE2PO4      Partition coefficient for phosphate                **
**                 in layer 2                               L/kg      **
**    kdpie1       Variable that determines salinity effect on        **
**                 DPIE1PO4                                  ppt      **
**    kpie2        Variable that determines salinity effect on        **
**                 PIE2PO4                                  ppt       **
**    O2CRIT       Critical dissolved oxygen concentration for        **
**                 layer 1 incremental phosphate sorption   mg O2/L   **
**    KMO2DP       Particle mixing half saturation constant           **
**                 for oxygen                               mg O2/L   **
**    TEMPBEN      Temperature at which benthic stress                **
**                 accumulation is reset to zero            deg C     **
**    KBENSTR      Decay constant for benthic stress        /day      **
**    KLBNTH       Ratio of bio-irrigation to bioturbation            **
**    DPMIN        Minimum particle diffusion coefficient   m2/day    **
**    KAPPCH4      methane oxidation reaction velocity      m/day     **
**    THTACH4      theta for methane oxidation                        **
**                                                                    **
** Output                                                             **
**                                                                    **
**    The subroutine returns fluxes for                               **
**                                                                    **
**     JSOD, JAQSOD, JCH4AQ and JCH4G  [gm o2*/m2-day]                **
**     JNH4, JPO4, JNO3          [mg/m2-day]                          **
**                                                                    **
**    via array BFLUX in COMMON /BENTHC/                              **
**                                                                    **
************************************************************************

      MODULE SED
      USE WQM;  USE FILE_INFO
      use sediment_mod
!.... PARALLEL SECTION BEGINS
#ifdef PARALLEL
      USE MESSENGER    
!.... PARALLEL SECTION ENDS
#endif
      IMPLICIT NONE
      INTEGER,SAVE  :: INTSEDC, IWC, IERR, ITEMP
      REAL,SAVE     :: HSEDALL, DIFFT, XKBO2, ALPHB, CCHLB
      REAL,SAVE     :: FLX1WC, FLX2WC, FLX3WC, FLX4WC
      REAL,SAVE     :: FLX5WC, FLX6WC, FLX7WC, FLX8WC
      REAL,SAVE     :: STP20, DF, PO4AVL, BFOR, W12MIN
      REAL,SAVE     :: DOLOW, LOGICT, IK
      REAL,SAVE     :: XKPOP1, XKPOP2, XKPOP3, XKPON1, XKPON2, XKPON3
      REAL,SAVE     :: XKPOC1, XKPOC2, XKPOC3, XKPIP
      REAL,SAVE     :: RDD, RMORT, XPOC1LIM, TEMP, XPOC2LIM, DOH2
      REAL,SAVE     :: FRDOM, FRG1M, FRG2M
      REAL,SAVE     :: ERROR, A1, A2
      REAL,SAVE     :: XJCNO31, FP1SO4, FP2SO4, HS2AV, XJ2, XJ2CH4, X1J2
      REAL,SAVE     :: PF, PIOP, FTB, PRNB, FRDOB, FRDOCB
      REAL,SAVE     :: FLUXHS, FLUXHSCH4, VJCH4G
      
      CONTAINS


********************************************************************************
**                    S U B R O U T I N E   S E D _ R E A D                   **
********************************************************************************
      SUBROUTINE SED_READ
      USE WQM;  USE WQM_INIT; USE FILE_INFO
      IMPLICIT NONE
! added next line 10/17/05
      SAVE    

***** Variable declarations

      CHARACTER FRNAME(14)*24, BALC*3, SPVARS*8, SPVARLR*8, SPVARB*8,
     $          PRINTS*8, PRINTLR*8, PRINTB*8, SALTEFF*8
      INTEGER   I, JG, JT

***** Data declarations

      DATA FRNAME
     .     /'Group 1 algal phosphorus', 'Group 2 algal phosphorus',
     .      'Group 3 algal phosphorus', 'Detrital org phosphorus ',
     .      'Group 1 algal nitrogen  ', 'Group 2 algal nitrogen  ',
     .      'Group 3 algal nitrogen  ', 'Detrital org nitrogen   ',
     .      'Group 1 algal carbon    ', 'Group 2 algal carbon    ',
     .      'Group 3 algal carbon    ', 'Benthic algal carbon    ',
     .      'Benthic algal nitrogen  ', 'Benthic algal phosphorus'/

********************************************************************************
**                                  Inputs                                    **
********************************************************************************

      READ(BFI,1000,ERR=10100)  HSEDALL, INTSEDC
      READ(BFI,1010,ERR=10100)  DIFFT
      READ(BFI,1010,ERR=10100)  SALTSW, SALTND
      READ(BFI,1010,ERR=10100)  FRPPH1
      READ(BFI,1010,ERR=10100)  FRPPH2
      READ(BFI,1010,ERR=10100)  FRPPH3
      READ(BFI,1010,ERR=10100)  FRPPHB    
      READ(BFI,1010,ERR=10100)  FRNPH1
      READ(BFI,1010,ERR=10100)  FRNPH2
      READ(BFI,1010,ERR=10100)  FRNPH3
      READ(BFI,1010,ERR=10100)  FRNPHB
      READ(BFI,1010,ERR=10100)  FRCPH1
      READ(BFI,1010,ERR=10100)  FRCPH2
      READ(BFI,1010,ERR=10100)  FRCPH3
      READ(BFI,1010,ERR=10100)  FRCPHB
      READ(BFI,1010,ERR=10100) (KPDIAG(JG),DPTHTA(JG),JG=1,3)
      READ(BFI,1010,ERR=10100) (KNDIAG(JG),DNTHTA(JG),JG=1,3)
      READ(BFI,1010,ERR=10100) (KCDIAG(JG),DCTHTA(JG),JG=1,3)
      READ(BFI,1015,ERR=10100)  M1,M2,THTADP,THTADD
      READ(BFI,1015,ERR=10100)  KAPPNH4F,KAPPNH4S,PIENH4,THTANH4,KMNH4,
     .                          KMNH4O2
      READ(BFI,1015,ERR=10100)  k1no3,K2NO3,THTANO3
      READ(BFI,1015,ERR=10100)  KAPPD1,KAPPP1,PIE1S,PIE2S,THTAPD1,
     .                          KMHSO2
      READ(BFI,1015,ERR=10100)  O2CRIT,KMO2DP
      READ(BFI,1015,ERR=10100)  TEMPBEN,KBENSTR,KLBNTH,DPMIN
      READ(BFI,1015,ERR=10100)  KAPPCH4,THTACH4,KMCH4O2,KMSO4

c benthic algae 

      READ(BFI,1012,ERR=10100)  BALC
      READ(BFI,1015,ERR=10100)  PMB, ANCB, APCB, KTGB1, KTGB2, TMB
      READ(BFI,1015,ERR=10100)  ALPHB, CCHLB, KESED, KEBALG, KHNB, 
     $                          KHPB
      READ(BFI,1015,ERR=10100)  BMRB, BPRB, KTBB, TRB, BALGMIN
      READ(BFI,1015,ERR=10100)  FNIB, FPIB, FRDOB, FRDOCB

C net settling rates

      READ(BFI,1060,ERR=10100)  SPVARS,PRINTS
      IF (SPVARS .EQ. 'CONSTANT') THEN
        READ(BFI,1020,ERR=10100) WSSBNET(1),WSLBNET(1),WSRBNET(1),
     .    WSG3BNET(1), WS1BNET(1),WS2BNET(1),WS3BNET(1),WSPIPBNET(1)
        DO BB=2,NBB
          WSSBNET(BB)=WSSBNET(1)
          WSLBNET(BB)=WSLBNET(1)
          WSRBNET(BB)=WSRBNET(1)
          WSG3BNET(BB)=WSG3BNET(1)
          WS1BNET(BB)=WS1BNET(1)
          WS2BNET(BB)=WS2BNET(1)
          WS3BNET(BB)=WS3BNET(1)
          WSPIPBNET(BB)=WSPIPBNET(1)
        END DO
      ELSE
        READ(BFI,1020,ERR=10100) (WSSBNET(BB),WSLBNET(BB),WSRBNET(BB),
     .    WSG3BNET(BB),WS1BNET(BB),WS2BNET(BB),WS3BNET(BB),
     .    WSPIPBNET(BB),BB=1,NBB)
      END IF

C burial and mixing rates

      READ(BFI,1060,ERR=10100)  SPVARB,PRINTB
      IF (SPVARB .EQ. 'CONSTANT') THEN
        READ(BFI,1022,ERR=10100) VSED(1),VPMIX(1),VDMIX(1)
        DO BB=2,NBB
          VSED(BB)=VSED(1)
          VPMIX(BB)=VPMIX(1)
          VDMIX(BB)=VDMIX(1)
        END DO
      ELSE
        READ(BFI,1022,ERR=10100) (VSED(BB),VPMIX(BB),VDMIX(BB),BB=1,NBB)
      END IF

C spatially-varying P sorption coefficients 

      READ(BFI,1062,ERR=10100)  SPVARLR,PRINTLR,SALTEFF
C Testa's values kpie2 = 1.5, kdpie1 = 3.6
      IF (SALTEFF .EQ. '      ON') READ(BFI,1015) kpie2, kdpie1   
      IF (SPVARLR .EQ. 'CONSTANT') THEN
        READ(BFI,1042,ERR=10100) PIE2PO4(1), DPIE1PO4(1)
        DO BB=2,NBB
          PIE2PO4(BB)  = PIE2PO4(1)
	  DPIE1PO4(BB) = DPIE1PO4(1)
        END DO
      ELSE        
        READ(BFI,1042,ERR=10100) (PIE2PO4(BB),DPIE1PO4(BB),
     .                          BB=1,NBB)
      END IF
 
***** Define logical variables

      STEADY_STATE_SED = INTSEDC.EQ.1
      BALGAE_CALC = BALC .EQ. ' ON'
      SALT_PART = SALTEFF .EQ. '      ON'

********************************************************************************
**                                 Outputs                                    **
********************************************************************************

      IF (BENTHIC_OUTPUT) THEN
        OPEN (BFO,FILE=BFOFN)
        WRITE(BFO,2000)
        WRITE(BFO,2020) HSEDALL
        IF (STEADY_STATE_SED) THEN
          WRITE(BFO,2022)
        ELSE
          WRITE(BFO,2025)
        END IF
        WRITE(BFO,2030)
        WRITE(BFO,2040)  SSNAME(1)
        WRITE(BFO,2050) (CTEMP(BB),BB=1,NBB)
        WRITE(BFO,2060)  SSNAME(2)
        WRITE(BFO,2070) ((CPOP(BB,JG),JG=1,3),BB=1,NBB)
        WRITE(BFO,2060)  SSNAME(3)
        WRITE(BFO,2070) ((CPON(BB,JG),JG=1,3),BB=1,NBB)
        WRITE(BFO,2060)  SSNAME(4)
        WRITE(BFO,2070) ((CPOC(BB,JG),JG=1,3),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(6)
        WRITE(BFO,2050) (PO4T2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(7)
        WRITE(BFO,2050) (NH4T2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(8)
        WRITE(BFO,2050) (NO3T2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(9)
        WRITE(BFO,2050) (HST2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(14)
        WRITE(BFO,2050) (BENSTR1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(10)
        WRITE(BFO,2050) (CH4T2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2040)  SSNAME(12)
        WRITE(BFO,2050) (SO4T2TM1S(BB),BB=1,NBB)
        WRITE(BFO,2080)  0.0001*DIFFT
        WRITE(BFO,2090)  SALTSW, SALTND, SALTLL, SALTUL
        WRITE(BFO,2100)
        WRITE(BFO,2110)  FRNAME(1),FRPPH1
        WRITE(BFO,2110)  FRNAME(2),FRPPH2
        WRITE(BFO,2110)  FRNAME(3),FRPPH3
        WRITE(BFO,2110)  FRNAME(14),FRPPHB
        WRITE(BFO,2110)  FRNAME(5),FRNPH1
        WRITE(BFO,2110)  FRNAME(6),FRNPH2
        WRITE(BFO,2110)  FRNAME(7),FRNPH3
        WRITE(BFO,2110)  FRNAME(13),FRNPHB
        WRITE(BFO,2110)  FRNAME(9),FRCPH1
        WRITE(BFO,2110)  FRNAME(10),FRCPH2
        WRITE(BFO,2110)  FRNAME(11),FRCPH3
        WRITE(BFO,2110)  FRNAME(12),FRCPHB
        WRITE(BFO,2120) (KPDIAG(JG),DPTHTA(JG),JG=1,3),(KNDIAG(JG),
     .                   DNTHTA(JG),JG=1,3),(KCDIAG(JG),DCTHTA(JG),
     .                   JG=1,3)
        WRITE(BFO,2252)  BALC
        WRITE(BFO,2242)  PMB, ANCB, APCB, KTGB1, KTGB2, TMB
        WRITE(BFO,2244)  ALPHB, CCHLB, KESED, KEBALG, KHNB, KHPB
        WRITE(BFO,2246)  BMRB, BPRB, KTBB, TRB, BALGMIN
        WRITE(BFO,2248)  FNIB, FPIB, FRDOB, FRDOCB
        WRITE(BFO,2170)  M1,M2,THTADP,THTADD
        WRITE(BFO,2180)  KAPPNH4F,KAPPNH4S,PIENH4,THTANH4,KMNH4,KMNH4O2
        WRITE(BFO,2190)  k1no3,K2NO3,THTANO3
        WRITE(BFO,2200)  KAPPD1,KAPPP1,PIE1S,PIE2S,THTAPD1,KMHSO2
        WRITE(BFO,2220)  O2CRIT,KMO2DP
        WRITE(BFO,2230)  TEMPBEN,KBENSTR,KLBNTH,DPMIN
        WRITE(BFO,2240)  KAPPCH4,THTACH4
        WRITE(BFO,2130) (BBN(BB),WSSBNET(BB),WSLBNET(BB),WSRBNET(BB),
     .                   WSG3BNET(BB),
     .                   WS1BNET(BB),WS2BNET(BB),WS3BNET(BB),
     .                   WSPIPBNET(BB),BB=1,NBB)
        WRITE(BFO,2140) (BBN(BB),VSED(BB),BB=1,NBB)
        WRITE(BFO,2150) (BBN(BB),VPMIX(BB),BB=1,NBB)
        WRITE(BFO,2160) (BBN(BB),VDMIX(BB),BB=1,NBB)
        CLOSE(BFO)
      END IF

********************************************************************************
**                             Initializations                                **
********************************************************************************

***** Convert cell heights and burial velocities to sediment units

      DIFFT = 0.0001*DIFFT
      DO 10000 BB=1,NBB
        HSED(BB) = HSEDALL*0.01
        VSED(BB) = VSED(BB)*2.73791E-5
10000 CONTINUE

***** Set sediment concentrations to initial concentrations

      DO 10010 BB=1,NBB
        POP1TM1S(BB) = CPOP(BB,1)
        POP2TM1S(BB) = CPOP(BB,2)
        POP3TM1S(BB) = CPOP(BB,3)
        PIPTM1S(BB)  = CPIP(BB)
        PON1TM1S(BB) = CPON(BB,1)
        PON2TM1S(BB) = CPON(BB,2)
        PON3TM1S(BB) = CPON(BB,3)
        POC1TM1S(BB) = CPOC(BB,1)
        POC2TM1S(BB) = CPOC(BB,2)
        POC3TM1S(BB) = CPOC(BB,3)
10010 CONTINUE

***** Set up reaction rates in table look-up form

      DO 10020 JT=1,350
        TEMP         = REAL(JT-1)/10.+0.05
        TEMP20       = TEMP-20.
        TEMP202      = TEMP20/2.
        ZHTANH4F(JT) = KAPPNH4F*THTANH4**TEMP202
        ZHTANH4S(JT) = KAPPNH4S*THTANH4**TEMP202
        ZHTAD1(JT)   = KAPPD1*THTAPD1**TEMP202
        ZHTAP1(JT)   = KAPPP1*THTAPD1**TEMP202
        ZHTA1NO3(JT) = k1no3*THTANO3**TEMP202
        ZHTA2NO3(JT) = K2NO3*THTANO3**TEMP20
        ZL12NOM(JT)  = THTADD**TEMP20
        ZW12NOM(JT)  = THTADP**TEMP20
        ZHTAPON1(JT) = KPON1*THTAPON1**TEMP20
        ZHTAPON2(JT) = KPON2*THTAPON2**TEMP20
        ZHTAPON3(JT) = KPON3*THTAPON3**TEMP20
        ZHTAPOC1(JT) = KPOC1*THTAPOC1**TEMP20
        ZHTAPOC2(JT) = KPOC2*THTAPOC2**TEMP20
        ZHTAPOC3(JT) = KPOC3*THTAPOC3**TEMP20
        ZHTAPOP1(JT) = KPOP1*THTAPOP1**TEMP20
        ZHTAPOP2(JT) = KPOP2*THTAPOP2**TEMP20
        ZHTAPOP3(JT) = KPOP3*THTAPOP3**TEMP20
        ZHTACH4(JT)  = KAPPCH4*THTACH4**TEMP202
10020 CONTINUE

***** Turn off settling

      IF (.NOT.SETTLING) THEN
        DO 10030 BB=1,NBB
          WSSBNET(BB) = 0.
          WSLBNET(BB) = 0.
          WSG3BNET(BB)= 0.
          WSRBNET(BB) = 0.
          WS1BNET(BB) = 0.
          WS2BNET(BB) = 0.
          WS3BNET(BB) = 0.
          WSPIPBNET(BB) = 0.
10030   CONTINUE
      END IF

***** Initialize accumulators for steady-state computations

      IF (STEADY_STATE_SED) THEN
        TINTIM = 0.
        DO 10035 BB=1,NBB
          AG3CFL(BB) = 0.
          AG3NFL(BB) = 0.
          AG3PFL(BB) = 0.
          ASDTMP(BB) = 0.
10035   CONTINUE
      END IF

***** Input FORMAT'S

 1000 FORMAT(:////8X,F8.0,I8)
 1002 format(/7f10.4)
 1010 FORMAT(8F10.0)
 1012 FORMAT(//13X,A3)
 1015 FORMAT(//8X,8F8.1)
 1020 FORMAT(:///(8X,8F8.1))
 1022 FORMAT(:///(8X,3F8.1))
 1030 FORMAT(I10)
 1040 FORMAT(:///(8X,6F8.1))
 1042 FORMAT(:///(8X,2F8.1))
 1060 FORMAT(//8X,2A8,F8.0)
 1062 FORMAT(//8X,3A8,F8.0)
 
***** Output FORMAT'S

 2000 FORMAT(///34X,'Sediment-water column linkages and sediment ',
     .       'depths and volumes'/)
 2020 FORMAT(/' ACTIVE LAYER DEPTH ',F8.3,' CM')
 2022 FORMAT(/' STEADY-STATE VALUES OF G3 COMPUTED'/)
 2025 FORMAT(/' NO STEADY-STATE VALUES OF G3 COMPUTED'/)
 2030 FORMAT(////33X,'S E D I M E N T   I N I T I A L   C O N D I T ',
     .       'I O N S'/)
 2040 FORMAT(//25X,'Sediment initial conditions for ',A20/)
 2050 FORMAT(13X,3(7X,1PE11.4))
 2060 FORMAT(//25X,'Sediment initial conditions for ',A20/
     .       37X,'G1',22X,'G2',22X,'G3'/)
 2070 FORMAT(18X,3(2X,1PE11.4))
 2080 FORMAT(//30X,'Temperature diffusion coefficient ',E10.3,
     .       ' cm**2/sec')
 2090 FORMAT(//31X,'If salinity < ',F10.3,' ppt, methane formed',/
     .      30X,'If salinity < ',F10.3,' ppt, high nit/denit used'/
     . 30x,'If ',F10.3,' < salinity < ',F10.3,' low sorption used')
 2100 FORMAT(//30X,'Particulate organic matter G-model splits'/
     .       10X,'fraction of....',5X,'recycled to',5X,'G1',5X,'G2',
     .       5X,'G3')
 2110 FORMAT(6X,A24,11X,3F7.2)
 2120 FORMAT(//30X,'Diagenesis rates (/day)  Temp corr factor'/
     .       30X,'Phosphorus'/
     .       39X,'G1',E11.3,5X,F7.3/
     .       39X,'G2',E11.3,5X,F7.3/
     .       39X,'G3',E11.3,5X,F7.3/
     .       30X,'Nitrogen'/
     .       39X,'G1',E11.3,5X,F7.3/
     .       39X,'G2',E11.3,5X,F7.3/
     .       39X,'G3',E11.3,5X,F7.3/
     .       30X,'Carbon'/
     .       39X,'G1',E11.3,5X,F7.3/
     .       39X,'G2',E11.3,5X,F7.3/
     .       39X,'G3',E11.3,5X,F7.3)
 2130 FORMAT(//3X,'BBN',6X,'WSSNET',4X,'WSLNET',4X,'WSRNET',4X,
     .        'WSG3NET',3X,'WS1NET',4X,'WS2NET',4X,'WS3NET',
     .         4X,'WSPIPNET'/(I7,7F10.3))
 2140 FORMAT(//31X,'Sedimentation rates (cm/yr)'/
     .       10X,8(I5,F6.2))
 2150 FORMAT(//30X,'Sediment solid-phase mixing rates (m**2/day)'/
     .       10X,8(I5,F6.2))
 2160 FORMAT(//30X,'Sediment dissolved-phase mixing rates (m**2/day)'/
     .       10X,8(I5,F6.2))
 2170 FORMAT(//35X,'Additional constants'/
     .       30X,'M1........',F8.2,' kg/l'/
     .       30X,'M2........',F8.2,' kg/l'/
     .       30X,'THTADP....',F8.3,/
     .       30X,'THTADD....',F8.3)
 2180 FORMAT(30X,'KAPPNH4F..',F8.3,' m/day'/
     .       30X,'KAPPNH4S..',F8.3,' m/day'/
     .       30X,'PIENH4....',F8.3,' l/kg'/
     .       30X,'THTANH4...',F8.3,/
     .       30X,'KMNH4.....',F8.3,' mg n/m**3'/
     .       30X,'KMNH4O2...',F8.3,' mg o2/l')
 2190 FORMAT(30X,'k1no3.....',F8.3,' m/day'/
     .       30X,'K2NO3.....',F8.3,' /day'/
     .       30X,'THTANO3...',F8.3)
 2200 FORMAT(30X,'KAPPD1....',F8.3,' m/day'/
     .       30X,'KAPPP1....',F8.3,' m/day'/
     .       30X,'PIE1S.....',F8.3,' l/kg'/
     .       30X,'PIE2S.....',F8.3,' l/kg'/
     .       30X,'THTAPD1...',F8.3,/
     .       30X,'KMHSO2....',F8.3,' mg o2/l')
 2220 FORMAT(30X,'O2CRIT....',F8.3,' mg o2/l'/
     .       30X,'KMO2DP....',F8.3,' mg o2/l')
 2230 FORMAT(30X,'TEMPBEN...',F8.3,' deg c'/
     .       30X,'KBENSTR...',F8.3,' /day'/
     .       30X,'KLBNTH....',F8.3,'---'/
     .       30X,'DPMIN.....',F8.3,' m2/d')
 2240 FORMAT(30X,'KAPPCH4...',F8.3,' m/day'/
     .       30X,'THTACH4...',F8.3)
 2242 FORMAT(' PMB = ',F10.3/' ANCB = ',F10.3/' APCB = ',F10.3/
     $  ' KTGB1 = ',F10.3/' KTGB2 = ',F10.3/' TMB = ',F10.3)
 2244 FORMAT(' ALPHB = ',F10.3/' CCHLB = ',F10.3/
     $ ' KESED = ',F10.3/' KEBALG = ',F10.3,
     $ ' KHNB = ',F10.3/' KHPB = ',F10.3)
 2246 FORMAT(' BMRB = ',F10.3/' BPRB = ',F10.3/' KTBB = ', F10.3/
     $  ' TRB = ',F10.3/ ' BALGMIN = ',F10.3)
 2248 FORMAT(' FNIB = ',F10.3/' FPIB = ',F10.3/' FRDOB = ',F10.3/
     $  ' FRDOCB = ',F10.3)
 2252 FORMAT(/' BENTHIC ALGAE ARE ',A3)

***** Error output FORMAT's

 3000 FORMAT(///5X,'Zbrent failure, ierr=',I2,' in segment ',I5/
     .       5X,'Program termination follows diagnostic dumps')
 3010 FORMAT(/' Read error in sediment input deck')
      RETURN

***** Error traps

10100 CONTINUE
      IF (BENTHIC_OUTPUT) WRITE(BFO,3010)

!.... PARALLEL SECTION BEGINS
#ifdef PARALLEL
      CALL COMM_EXIT()
      STOP 'SED_READ'
#else
!.... PARALLEL SECTION ENDS
      STOP 'SED_READ'
#endif
      END SUBROUTINE SED_READ



********************************************************************************
**                    S U B R O U T I N E   S E D _ C A L C                   **
**                           Sediment Calculations                            **
********************************************************************************

      SUBROUTINE SED_CALC
      IMPLICIT NONE
! added next line 10/17/05
      SAVE    
      INTEGER I, J, JSF, N
      REAL :: SODMIN, SODMAX, SAVEFCT, PFD, dpie

***** Pass WQM time-step (in days) to sediment subr

      DLTS = DLT/86400.
      IF (STEADY_STATE_SED) TINTIM = TINTIM+DLTS

******* Assign base net settling rates

      DO BB=1,NBB
        WSSNET(BB) = WSSBNET(BB)
        WSLNET(BB) = WSLBNET(BB)
        WSRNET(BB) = WSRBNET(BB)
        WSG3NET(BB)= WSG3BNET(BB)
        WS1NET(BB) = WS1BNET(BB)
        WS2NET(BB) = WS2BNET(BB)
        WS3NET(BB) = WS3BNET(BB)
        WSPIPNET(BB) = WSPIPBNET(BB)
      END DO

******* Adjust net settling for SAV effect

cvjp modified 11/3/2005
        IF (SAV_CALC) THEN
          DO I=1,NSAVCELL
            B=SAVCELL(I)
              SAVEFCT   = (LEAF(B)  + STEM(B)) * SAVFRAC(B)
              WSSNET(B) = WSSNET(B) + WSSSAV * SAVEFCT
              WSLNET(B) = WSLNET(B) + WSLSAV * SAVEFCT
              WSRNET(B) = WSRNET(B) + WSRSAV * SAVEFCT
              WSG3NET(B)= WSG3NET(B)+ WSG3SAV* SAVEFCT
              WS1NET(B) = WS1NET(B) + WS1SAV * SAVEFCT
              WS2NET(B) = WS2NET(B) + WS2SAV * SAVEFCT
              WS3NET(B) = WS3NET(B) + WS3SAV * SAVEFCT
              WSPIPNET(B) = WSPIPNET(B) + WSPIPSAV * SAVEFCT
          END DO
        END IF

***** Calculate fluxes

      DO 10060 BB=1,NBB
        IWC = BBN(BB)

******* Flux rate

        FLX1WC    = 1000.*WSSNET(BB)
        FLX2WC    = 1000.*WSLNET(BB)
        FLX3WC    = 1000.*WSRNET(BB)
        FLX7WC    = 1000.*WSG3NET(BB)
        FLX4WC    = 1000.*WS1NET(BB)
        FLX5WC    = 1000.*WS2NET(BB)
        FLX6WC    = 1000.*WS3NET(BB)
        FLX8WC    = 1000.*WSPIPNET(BB)

C       *** Fluxes

        FLXPOP(BB,1) = FLX4WC*APC1*FRPPH1(1)*B1(IWC)
     .                +FLX5WC*APC2*FRPPH2(1)*B2(IWC)
     .                +FLX6WC*APC3*FRPPH3(1)*B3(IWC)
     .                +FLX2WC*LPOP(IWC)
     .                +SFLUXP(BB)*FRPPH3(1)       ! suspension feeders
        FLXPOP(BB,2) = FLX4WC*APC1*FRPPH1(2)*B1(IWC)
     .                +FLX5WC*APC2*FRPPH2(2)*B2(IWC)
     .                +FLX6WC*APC3*FRPPH3(2)*B3(IWC)
     .                +FLX3WC*RPOP(IWC)
     .                +SFLUXP(BB)*FRPPH3(2)       ! suspension feeders
        FLXPOP(BB,3) = FLX4WC*APC1*FRPPH1(3)*B1(IWC)
     .                +FLX5WC*APC2*FRPPH2(3)*B2(IWC)
     .                +FLX6WC*APC3*FRPPH3(3)*B3(IWC)
     .                +FLX7WC*G3POP(IWC)
     .                +SF_RPOP(BB)                ! suspension feeders
     .                +SFLUXP(BB)*FRPPH3(3)       ! suspension feeders
        FLXPIP(BB)   = FLX8WC*PIP(IWC)
     .                +SF_PIP(BB)                 ! suspension feeders
        FLXPON(BB,1) = FLX4WC*ANC1*FRNPH1(1)*B1(IWC)
     .                +FLX5WC*ANC2*FRNPH2(1)*B2(IWC)
     .                +FLX6WC*ANC3*FRNPH3(1)*B3(IWC)
     .                +FLX2WC*LPON(IWC)
     .                +SFLUXN(BB)*FRNPH3(1)       ! suspension feeders
        FLXPON(BB,2) = FLX4WC*ANC1*FRNPH1(2)*B1(IWC)
     .                +FLX5WC*ANC2*FRNPH2(2)*B2(IWC)
     .                +FLX6WC*ANC3*FRNPH3(2)*B3(IWC)
     .                +FLX3WC*RPON(IWC)
     .                +SFLUXN(BB)*FRNPH3(2)       ! suspension feeders
        FLXPON(BB,3) = FLX4WC*ANC1*FRNPH1(3)*B1(IWC)
     .                +FLX5WC*ANC2*FRNPH2(3)*B2(IWC)
     .                +FLX6WC*ANC3*FRNPH3(3)*B3(IWC)
     .                +FLX7WC*G3PON(IWC)
     .                +SF_RPON(BB)                ! suspension feeders
     .                +SFLUXN(BB)*FRNPH3(3)       ! suspension feeders
        FLXPOC(BB,1) = FLX4WC*FRCPH1(1)*B1(IWC)
     .                +FLX5WC*FRCPH2(1)*B2(IWC)
     .                +FLX6WC*FRCPH3(1)*B3(IWC)
     .                +FLX2WC*LPOC(IWC)
     .                +SFLUXC(BB)*FRCPH3(1)       ! suspension feeders
        FLXPOC(BB,2) = FLX4WC*FRCPH1(2)*B1(IWC)
     .                +FLX5WC*FRCPH2(2)*B2(IWC)
     .                +FLX6WC*FRCPH3(2)*B3(IWC)
     .                +FLX3WC*RPOC(IWC)
     .                +SFLUXC(BB)*FRCPH3(2)       ! suspension feeders
        FLXPOC(BB,3) = FLX4WC*FRCPH1(3)*B1(IWC)+
     .                 FLX5WC*FRCPH2(3)*B2(IWC)
     .                +FLX6WC*FRCPH3(3)*B3(IWC)
     .                +FLX7WC*G3POC(IWC)
     .                +SF_RPOC(BB)                ! suspension feeders
     .                +SFLUXC(BB)*FRCPH3(3)       ! suspension feeders

C       *** Sum particulate fluxes to sediments, negative into sediments

        PPFWS(BB)=-0.001*(FLXPOP(BB,1)+FLXPOP(BB,2)+FLXPOP(BB,3))
C CLAY, SILT, SAND, DO NOT APPEAR TO BE CONSIDERED HERE.
	IF (SEDKIN .EQ. 'SSI') THEN
          PF    = KADPO4*SSI(IWC)/(1.+KADPO4*SSI(IWC))
          PPFWS(BB)  =  PPFWS(BB)-0.001*FLX1WC*PF*PO4(IWC)
        END IF
        PNFWS(BB)=-0.001*(FLXPON(BB,1)+FLXPON(BB,2)+FLXPON(BB,3))
        PCFWS(BB)=-0.001*(FLXPOC(BB,1)+FLXPOC(BB,2)+FLXPOC(BB,3))
	PIPFWS(BB)= -0.001*FLXPIP(BB)
        SSFWS(BB)=-WSSNET(BB)*SSI(IWC)-0.001*SF_SSI(BB)

10060 CONTINUE

C ADD IN THE FLUX FROM ROOT MORTALITY

cvjp modified 11/3/2005
        IF (SAV_CALC) THEN
          DO I=1,NSAVCELL
            B=SAVCELL(I)
            DO J=1,3
              FLXPOC(B,J) = FLXPOC(B,J)+1000.*SEDCSAV(B)*FRCSAV(J)
              FLXPON(B,J) = FLXPON(B,J)+1000.*SEDNSAV(B)*FRNSAV(J)
              FLXPOP(B,J) = FLXPOP(B,J)+1000.*SEDPSAV(B)*FRPSAV(J)
            END DO
          END DO
        END IF

C     *** Accumulate fluxes for steady-state computation

      IF (STEADY_STATE_SED) THEN
        DO 10050 BB=1,NBB
          AG3CFL(BB) = AG3CFL(BB)+FLXPOC(BB,3)*DLTS
          AG3NFL(BB) = AG3NFL(BB)+FLXPON(BB,3)*DLTS
          AG3PFL(BB) = AG3PFL(BB)+FLXPOP(BB,3)*DLTS
10050   CONTINUE
      ENDIF

***** Assign previous timestep concentrations to particulate organics

      DO 10070 BB=1,NBB
        CPOP(BB,1) = POP1TM1S(BB)
        CPOP(BB,2) = POP2TM1S(BB)
        CPOP(BB,3) = POP3TM1S(BB)
        CPIP(BB)   = PIPTM1S(BB)
        CPON(BB,1) = PON1TM1S(BB)
        CPON(BB,2) = PON2TM1S(BB)
        CPON(BB,3) = PON3TM1S(BB)
        CPOC(BB,1) = POC1TM1S(BB)
        CPOC(BB,2) = POC2TM1S(BB)
        CPOC(BB,3) = POC3TM1S(BB)
10070 CONTINUE

***** Update sediment concentrations

      DO 10080 BB=1,NBB

******* Assign previous timestep concentrations

        NH41TM1  = NH41TM1S(BB)
        NO31TM1  = NO31TM1S(BB)
        HS1TM1   = HS1TM1S(BB)
        PO41TM1  = PO41TM1S(BB)
        BENSTR1  = BENSTR1S(BB)
        NH4T2TM1 = NH4T2TM1S(BB)
        NO3T2TM1 = NO3T2TM1S(BB)
        HST2TM1  = HST2TM1S(BB)
        PO4T2TM1 = PO4T2TM1S(BB)
        PON1TM1  = PON1TM1S(BB)
        PON2TM1  = PON2TM1S(BB)
        PON3TM1  = PON3TM1S(BB)
        POC1TM1  = POC1TM1S(BB)
        POC1     = POC1TM1
        POC2TM1  = POC2TM1S(BB)
        POC3TM1  = POC3TM1S(BB)
        POP1TM1  = POP1TM1S(BB)
        POP2TM1  = POP2TM1S(BB)
        POP3TM1  = POP3TM1S(BB)
        PIPTM1   = PIPTM1S(BB)
        CH4T2TM1 = CH4T2TM1S(BB)           ! CH4
        CH41TM1  = CH41TM1S(BB)            ! CH4
        SO4T2TM1 = SO4T2TM1S(BB)           ! CH4


C ACCOUNT FOR SAV NUTRIENT UPTAKE
C NOTE THIS IS OVER ALL CELLS, NOT JUST SAV CELLS SO SEDNH4SAV
C MUST BE ZEROED OUT FOR ALL CELLS

        IF (SAV_CALC) THEN
          NH4T2TM1 = NH4T2TM1 - 1000.*SEDNH4SAV(BB)*DLTS/HSED(BB)
          PO4T2TM1 = PO4T2TM1 - 1000.*SEDPO4SAV(BB)*DLTS/HSED(BB)
        END IF
        
        BFORMAX  = BFORMAXS(BB)
        ISWBEN   = ISWBENS(BB)
        H2       = HSED(BB)

******* Sedimentation, mixing rates, and sediment temperature

        W2    = VSED(BB)
        DP    = VPMIX(BB)
        DD    = VDMIX(BB)
        TEMPD = CTEMP(BB)
        STP20 = TEMPD-20.

******* Convert overlying water column concentrations into mg/m**3

        IWC  = BBN(BB)
	IF (SEDKIN .EQ. 'SSI') THEN
          DF   = 1./(1.+KADPO4*SSI(IWC))
	ELSE
          DF   = 1./(1.+KADPO4*SEDCLY(IWC))
	END IF
        PO4AVL = DF*PO4(IWC)
        PO40 = PO4AVL*1000.
        NH40 = NH4(IWC)*1000.
        NO30 = NO3(IWC)*1000.
	IF (SEDKIN .EQ. 'SSI') THEN	
          DF   = 1./(1.+KADSA*SSI(IWC))
	ELSE
          DF   = 1./(1.+KADSA*SEDCLY(IWC))
	END IF
        O20  = AMAX1(DO(IWC),0.010)
        HS0  = COD(IWC)
        SAL  = SALT(IWC)

C       Regression function to get SO4 concentration from SAL
C       [SO4] = 20 mg/L          for        [Cl] < 6 mg/L
C             = (10/3)[Cl]       for        [Cl] > 6 mg/L
C       1 ppt = 607.445 mg/L Cl

        IF (SAL .GT. 0.0099) THEN
           SO40MG = 20.0 + (27./190.)*607.445*SAL
        ELSE
           SO40MG = 20.0
        ENDIF

******* Methane saturation

        CH4SAT = 99.*(1.+(ZD(IWC)+BL(IWC,3)+HSED(BB))/10.)
     .           *0.9759**STP20

******* Evaluate the temperature dependent coefficients

        ITEMP    = 10.*TEMPD+1

******* Salinity dependence of nitrification and denitrification

        IF (SAL.LE.SALTND) THEN
          XAPPNH4  = ZHTANH4F(ITEMP)
        ELSE
          XAPPNH4  = ZHTANH4S(ITEMP)
        END IF
        XAPP1NO3 = ZHTA1NO3(ITEMP)
        XAPPD1   = ZHTAD1(ITEMP)
        XAPPP1   = ZHTAP1(ITEMP)
        XK2NO3   = ZHTA2NO3(ITEMP)*H2
        XAPPCH4  = ZHTACH4(ITEMP)
        KL12NOM  = DD/H2*ZL12NOM(ITEMP)
        W12NOM   = DP/H2*ZW12NOM(ITEMP)*POC1/1.0E5
        IF (ISWBEN.EQ.0) THEN
          IF (TEMPD.GE.TEMPBEN) THEN
            ISWBEN  = 1
            BFORMAX = 0.
          ENDIF
          BFOR = KMO2DP/(KMO2DP+O20)
        ELSE
          IF (TEMPD.LT.TEMPBEN) THEN
            ISWBEN = 0
          ENDIF
          BFORMAX = AMAX1(KMO2DP/(KMO2DP+O20),BFORMAX)
          BFOR    = BFORMAX
        ENDIF
        BENSTR = (BENSTR1+DLTS*BFOR)/(1.+KBENSTR*DLTS)
c## -- add minimum mixing term and bio-irrigation formulation
c##
c##     W12    = W12NOM*(1.-KBENSTR*BENSTR)
c##     KL12   = KL12NOM
c## -- w12min= Dpmin/h2 is minimum particle mixing
        W12MIN = DPMIN/H2
        W12    = W12NOM*(1.-KBENSTR*BENSTR)+W12MIN
c## -- klbnth is ratio of bio-irrigation to bio-particle mixing
        KL12   = KL12NOM + KLBNTH*W12

******* Lookup reaction rates

        ITEMP  = 10.*TEMPD+1
        XKPOP1 = ZHTAPOP1(ITEMP)*H2
        XKPOP2 = ZHTAPOP2(ITEMP)*H2
        XKPOP3 = ZHTAPOP3(ITEMP)*H2
	XKPIP  = 0.0                ! NON-REACTIVE FOR NOW
        XKPON1 = ZHTAPON1(ITEMP)*H2
        XKPON2 = ZHTAPON2(ITEMP)*H2
        XKPON3 = ZHTAPON3(ITEMP)*H2
        XKPOC1 = ZHTAPOC1(ITEMP)*H2
        XKPOC2 = ZHTAPOC2(ITEMP)*H2
        XKPOC3 = ZHTAPOC3(ITEMP)*H2

******* Calculate sediment concentrations

        DOH2=DLTS/H2

        PON1 = (FLXPON(BB,1)*DOH2+PON1TM1)/(1.+(XKPON1+W2)*DOH2)

        PON2 = (FLXPON(BB,2)*DOH2+PON2TM1)/(1.+(XKPON2+W2)*DOH2)

        PON3 = (FLXPON(BB,3)*DOH2+PON3TM1)/(1.+(XKPON3+W2)*DOH2)

        POC1 = (FLXPOC(BB,1)*DOH2+POC1TM1)/(1.+(XKPOC1+W2)*DOH2)

        POC2 = (FLXPOC(BB,2)*DOH2+POC2TM1)/(1.+(XKPOC2+W2)*DOH2)

        POC3 = (FLXPOC(BB,3)*DOH2+POC3TM1)/(1.+(XKPOC3+W2)*DOH2)

        POP1 = (FLXPOP(BB,1)*DOH2+POP1TM1)/(1.+(XKPOP1+W2)*DOH2)

        POP2 = (FLXPOP(BB,2)*DOH2+POP2TM1)/(1.+(XKPOP2+W2)*DOH2)

        POP3 = (FLXPOP(BB,3)*DOH2+POP3TM1)/(1.+(XKPOP3+W2)*DOH2)

        SPIP =  0.0 !(FLXPIP(BB)  *DOH2+PIPTM1) /(1.+(XKPIP +W2)*DOH2)

******* Assign diagenesis values for sediment model

        XJN = XKPON1*PON1+XKPON2*PON2+XKPON3*PON3
        XJC = XKPOC1*POC1+XKPOC2*POC2+XKPOC3*POC3
        XJP = XKPOP1*POP1+XKPOP2*POP2+XKPOP3*POP3

C TEMPORARY BYPASS OF FLUX ALGORITHMS
C        GO TO 66666
        
******* Evaluate the NH4, NO3, and SOD equations

        SOD = ZBRENT(IERR)

c       IF (IERR.NE.0.AND.BENTHIC_OUTPUT) WRITE(BFO,3000) IERR,BB

        if(ierr.ne.0)  then
         sodmin =   0.0001
         sodmax = 100.
         write(6,9000)   jday,ierr,bb,sal,so40mg,
     .                  (sfeed(bb,jsf),jsf=1,3),sodmin,sodmax
         write(6,9911) csodhs, csodch4, csod
 9911    format(/1x,' csodhs, csodch4, csod'/3e10.3)
         write(6,9910)  ch41,ch42,hst1,hs1,hs2
 9910    format(/1x,' ch41   ch42   hst1   hs1   hs2'/5e10.3)
         if(ierr.eq.2)  then
         write(6,9900)  jday,ctemp(bb),pop1,pop2,pop3
         write(6,9901)  pon1,pon2,pon3,poc1,poc2,poc3
         write(6,9902)  po4t2,hst2
         write(6,9903)
     .        (flxpop(bb,1)+flxpop(bb,2)+flxpop(bb,3))
     .       ,(flxpon(bb,1)+flxpon(bb,2)+flxpon(bb,3))
     .       ,(flxpoc(bb,1)+flxpoc(bb,2)+flxpoc(bb,3))
         write(6,9904)  o20,csod,sod,s
     .                 ,h2,HSED(bb),VSED(bb)
         write(6,9905)  xjp,xjn,xjc,jo2nh4,xjc1
         write(6,9906)  jpo4,jnh4,jno3,jhs,jch4aq,jch4g,benstr
         write(6,9907)  po40,po41,po42,po4t2,nh40,nh41,nh42,nh4t2
         write(6,9908)  no30,no31,no32,no3t2,hs1,hs2,hst2
         STOP
         else
         error=sedf(sodmin)
         write(6,9889)  jday,sodmin,error
         write(6,9900)  jday,ctemp(bb),pop1,pop2,pop3
         write(6,9901)  pon1,pon2,pon3,poc1,poc2,poc3
         write(6,9902)  po4t2,hst2
         write(6,9903)
     .        (flxpop(bb,1)+flxpop(bb,2)+flxpop(bb,3))
     .       ,(flxpon(bb,1)+flxpon(bb,2)+flxpon(bb,3))
     .       ,(flxpoc(bb,1)+flxpoc(bb,2)+flxpoc(bb,3))
         write(6,9904)  o20,csod,sod,s
     .                 ,h2,HSED(bb),VSED(bb)
         write(6,9911) csodhs, csodch4, csod
         write(6,9905)  xjp,xjn,xjc,jo2nh4,xjc1
         write(6,9906)  jpo4,jnh4,jno3,jhs,jch4aq,jch4g,benstr
         write(6,9907)  po40,po41,po42,po4t2,nh40,nh41,nh42,nh4t2
         write(6,9908)  no30,no31,no32,no3t2,hs1,hs2,hst2
         error=sedf(sodmax)
         write(6,9889)  jday,sodmax,error
         write(6,9900)  jday,ctemp(bb),pop1,pop2,pop3
         write(6,9901)  pon1,pon2,pon3,poc1,poc2,poc3
         write(6,9902)  po4t2,hst2
         write(6,9903)
     .        (flxpop(bb,1)+flxpop(bb,2)+flxpop(bb,3))
     .       ,(flxpon(bb,1)+flxpon(bb,2)+flxpon(bb,3))
     .       ,(flxpoc(bb,1)+flxpoc(bb,2)+flxpoc(bb,3))
         write(6,9904)  o20,csod,sod,s
     .                 ,h2,HSED(bb),VSED(bb)
         write(6,9911) csodhs, csodch4, csod
         write(6,9905)  xjp,xjn,xjc,jo2nh4,xjc1
         write(6,9906)  jpo4,jnh4,jno3,jhs,jch4aq,jch4g,benstr
         write(6,9907)  po40,po41,po42,po4t2,nh40,nh41,nh42,nh4t2
         write(6,9908)  no30,no31,no32,no3t2,hs1,hs2,hst2
         STOP
         endif
        endif
 9889    format(/5x,'Zbrent diagnostics at time =',f8.3,
     .     ' for sod =',f8.4,' error =',e12.3/)
 9000    format(/
     .      5x,'Zbrent failure at time =',f8.3,' with ierr=',i2/
     .      5x,'in sediment segment ir=',i5/
     .      5x,'with Salt, SO40mg     =',2e10.3/
     .      5x,' Sfeed=',3f11.3/
     .      5x,'(sodmin,sodmax=',f6.3,f6.3,')'/
     .      5x,'Program termination follows diagnostic dumps')
 9900    format(/1x,' time,ctemp,pop1,pop2,pop3'/8e10.3)
 9901    format(/1x,' pon1,pon2,pon3,poc1,poc2,poc3'/
     .          8e10.3)
 9902    format(/1x,' po4t2,hst2'/8e10.3)
 9903    format(/1x,' flxpop,flxpon,flxpoc'/8e10.3)
 9904    format(/1x,' o20,csod,sod,s,h2'
     .          ,',hsed,vsed'/10e10.3)
 9905    format(/1x,' jp,jn,jc,jo2nh4,xjc1'/8e10.3)
 9906    format(/1x,' jpo4,jnh4,jno3,jhs,jch4aq,jch4g,benstr'/
     .          7e10.3)
 9907    format(/1x,' po40,po41,po42,po4t2,nh40,nh41,nh42,nh4t2'/8e10.3)
 9908    format(/1x,' no30,no31,no32,no3t2,hs1,hs2,hst2'/8e10.3)

******* Accumulate remaining sums for steady-state computation

        IF (STEADY_STATE_SED) THEN
          ASDTMP(BB) = ASDTMP(BB)+TEMPD*DLTS
        END IF

******* Evaluate the PO4 equation

        K0H1D = 0.
        K0H1P = 0.
        KMC1  = 0.
        K1H1D = S
        K1H1P = 0.
        K2H2D = 0.
        K2H2P = 0.
        J1    = S*PO40
        K3    = 0.
	IF (SEDKIN .EQ. 'SSI') THEN
          PF    = KADPO4*SSI(IWC)/(1.+KADPO4*SSI(IWC))
          PIOP  = PF*PO4(IWC)
          J2    = XJP + FLX1WC*PIOP + SPO4*1000.*WSPO4(IWC)*PO4(IWC)
     $      + flxpip(bb)  ! DIRECT SETTLING OF PO4 03/19/09, add in pip 04/15/16 
	ELSE       
          PFD    = KADPO4*SEDCLY(IWC)*PO4(IWC)*WSED(1,1)
     $             /(1.+KADPO4*SEDCLY(IWC))  ! WSED = M/S      
          J2  = XJP + PFD*86.4E6    + SPO4*1000.*WSPO4(IWC)*PO4(IWC) 
     $      + flxpip(bb)     ! PFD = G/M2/S, add in pip 04/15/16
        END IF

******* Oxygen dependency of pie1

        PIE2 = pie2po4(bb)
        dpie = dpie1po4(bb)
        IF (SALT_PART) THEN
          PIE2 = PIE2 * (1.-salt(bb)/(salt(bb)+kpie2))
          dpie = dpie * (1.-salt(bb)/(salt(bb)+kdpie1))
        END IF
        IF (O20.LT.O2CRIT) THEN
          PIE1 = PIE2*DPIE**(O20/O2CRIT)
        ELSE
          PIE1 = PIE2*DPIE
        ENDIF
        CALL SEDTSFNL (PO41,PO42,PO4T1,PO4T2,PO41TM1,PO4T2TM1)
        JPO4 = S*(PO41-PO40)

******* Assign flux-flux results to wqm arrays

        ITEMP      = 10*TEMPD+1
        XAPP1NO3   = ZHTA1NO3(ITEMP)
        XK2NO3     = ZHTA2NO3(ITEMP)*H2
        BENDO(BB)  = -SOD
     .               - SODSF(BB)          ! suspension feeders
        MTVEL(BB)  = SOD/O20
        BENNH4(BB) = JNH4/1000.
     .             + JNH4SF(BB)/1000.     ! suspension feeders
        BENNO3(BB) = JNO3/1000.
        BENPO4(BB) = JPO4/1000.
     .             + JPO4SF(BB)/1000.     ! suspension feeders
        BENDOC(BB) = 0.0 + SF_LDOC(BB)/1000.
        BENCOD(BB) = JHS                  ! +JCH4AQ  CFC 040616
        BENCH4G(BB) = JCH4G 
        BENCH4A(BB) = JCH4AQ
        BENDEN(BB) = (XAPP1NO3*NO31+XK2NO3*NO32)/1000.

******* Fluxes due to burial of particulates

        BURIALN(BB) = (PON1+PON2+PON3+NO3T2+NH4T2)*W2
        BURIALP(BB) = (POP1+POP2+POP3+PO4T2+SPIP)*W2
        BURIALC(BB) = (POC1+POC2+POC3)*W2

******* Diagenesis of carbon forms

        DIAGENC(BB) = XJC/1000.

        IF (BALGAE_CALC) THEN

******* Benthic algae algorithms start here        

******* Calculate mean light in algal mat

          BLITE(BB) = IAVG(IWC)*EXP(-KESED)/(KEBALG+1.0E-8)/BBM(BB)
     $      *(1. - EXP(-(KEBALG+1.0E-8)*BBM(BB)))
        
******* Temperature effects

          IF (T(IWC).LT.TMB) THEN
            FTB = EXP(-KTGB1*(T(IWC)-TMB)**2)
          ELSE
            FTB = EXP(-KTGB2*(TMB-T(IWC))**2)
          END IF
        
******* Light effects

          IK = PMB*FTB/ALPHB
          FIB(BB) = BLITE(BB)/SQRT(IK*IK+BLITE(BB)*BLITE(BB)+1.0E-8)
        
******* Nutrient limitations

C COMPUTE AVAILABLE AMMONIUM AND NITRATE

          NH4AVL = BENNH4(BB)*DLTS + NH4(IWC)*BL(IWC,3)
          NH4AVL = MAX(0.,NH4AVL)
          NO3AVL = BENNO3(BB)*DLTS + NO3(IWC)*BL(IWC,3)
          NO3AVL = MAX(0.,NO3AVL)

C COMPUTE NITROGEN LIMITATION 

          NLB(BB) = (NH4AVL+NO3AVL)/(KHNB+NH4AVL+NO3AVL)

C COMPUTE NITROGEN PREFERENCE

          PRNB = NH4AVL*NO3AVL/((KHNB+NH4AVL)*(KHNB+NO3AVL))
     $    +NH4AVL*KHNB/((1.E-30+NH4AVL+NO3AVL)*(KHNB+NO3AVL))

******* Phosphorus available for algal growth

          IF (SEDKIN .EQ. 'SSI') THEN
            DF     = 1./(1.+KADPO4*SSI(IWC))
	  ELSE
            DF     = 1./(1.+KADPO4*SEDCLY(IWC))
	  END IF
          PO4AVL = DF*PO4(IWC)*BL(IWC,3)
          PO4AVL = PO4AVL + BENPO4(BB)*DLTS
          PO4AVL = MAX(0.,PO4AVL)
          PLB(BB) = PO4AVL/(KHPB+PO4AVL)

******* Base metabolism

C IF BIOMASS IS LESS THAN ALLOWED MINIMUM, SET METABOLISM TO ZERO

          IF (BBM(BB).GT.BALGMIN) THEN
            BMB(BB) = BMRB*EXP(KTBB*(T(IWC)-TRB))
          ELSE
            BMB(BB) = 0.
          END IF

******* Production

          PB(BB) = PMB*FTB*AMIN1(FIB(BB),NLB(BB),PLB(BB))/CCHLB

******* Net primary production
      
          NPPB(BB) = (PB(BB)-BMB(BB))*BBM(BB)                     
 
******* Predation

C IF BIOMASS IS LESS THAN ALLOWED MINIMUM, SET PREDATION TO ZERO

          IF (BBM(BB).GT.BALGMIN) THEN
            PRB(BB) = BBM(BB)*BPRB*EXP(KTBB*(T(IWC)-TRB))
          ELSE
            PRB(BB) = 0.
          END IF

C ADJUST PREDATION SO BIOMASS DOESN'T GO NEGATIVE

          PRB(BB) = MIN(PRB(BB),PB(BB)-BMB(BB)+0.99/DLTS)              

******* Compute effects of algal activity on benthic flux

          BANH4(BB) = ANCB *(BMB(BB)*FNIB - PRNB*PB(BB)+ PRB(BB)*FNIB)
     $              * BBM(BB) 
          BANO3(BB) = -(1. - PRNB) * PB(BB) * ANCB * BBM(BB)
          BAPO4(BB) = APCB *(BMB(BB)*FPIB - PB(BB)+ PRB(BB)*FPIB)
     $              * BBM(BB) 
          BADO(BB)  = ((1.3-0.3*PRNB)*PB(BB)-FRDOB*BMB(BB))
     $              * AOCR*BBM(BB)
          BADOC(BB) = (1.-FRDOB)*BMB(BB)*BBM(BB)
     $              + FRDOCB*PRB(BB)*BBM(BB)

c TEMPORARY FIX UP WHEN BENTHIC ALGAE ARE RUN WITHOUT DIAGENESIS     
c          BENNH4(BB) =  BANH4(BB)
c          BENNO3(BB) =  BANO3(BB) 
c          BENPO4(BB) =  BAPO4(BB)
c          BENDOC(BB) =  BADOC(BB)
c          BENDO(BB)  =  BADO(BB)  

          BENNH4(BB) = BENNH4(BB) + BANH4(BB)
          BENNO3(BB) = BENNO3(BB) + BANO3(BB) 
          BENPO4(BB) = BENPO4(BB) + BAPO4(BB)
          BENDOC(BB) = BENDOC(BB) + BADOC(BB)
          BENDO(BB)  = BENDO(BB)  + BADO(BB)  

C COMPUTE EFFECTS OF ALGAL ACTIVITY ON ORGANIC PARTICULATES (MG/M**3)

          BAPOC(BB) = (1.-FRDOCB)*PRB(BB)*BBM(BB)
          BAPON(BB) = ANCB*(1.-FNIB)*(BMB(BB)+PRB(BB))*BBM(BB)
          BAPOP(BB) = APCB*(1.-FPIB)*(BMB(BB)+PRB(BB))*BBM(BB)
          
          POC1 = POC1 + 1000. * BAPOC(BB)*FRCPHB(1)*DLTS/H2
          POC2 = POC2 + 1000. * BAPOC(BB)*FRCPHB(2)*DLTS/H2
          POC3 = POC3 + 1000. * BAPOC(BB)*FRCPHB(3)*DLTS/H2
          PON1 = PON1 + 1000. * BAPON(BB)*FRNPHB(1)*DLTS/H2
          PON2 = PON2 + 1000. * BAPON(BB)*FRNPHB(2)*DLTS/H2
          PON3 = PON3 + 1000. * BAPON(BB)*FRNPHB(3)*DLTS/H2
          POP1 = POP1 + 1000. * BAPOP(BB)*FRPPHB(1)*DLTS/H2
          POP2 = POP2 + 1000. * BAPOP(BB)*FRPPHB(2)*DLTS/H2
          POP3 = POP3 + 1000. * BAPOP(BB)*FRPPHB(3)*DLTS/H2

******* Accumulate fluxes for steady-state computation

        IF (STEADY_STATE_SED) THEN
        
          AG3CFL(BB) = AG3CFL(BB)+1000.*PRB(BB)*FRCPHB(3)*BBM(BB)*DLTS
          AG3NFL(BB) = AG3NFL(BB)+
     $                 1000. * PRB(BB)*FRNPHB(3)*ANCB*BBM(BB)*DLTS
          AG3PFL(BB) = AG3PFL(BB)+
     $                 1000. * PRB(BB)*FRPPHB(3)*APCB*BBM(BB)*DLTS
     
        END IF
        
******* Change in benthic algal biomass

          BBM(BB) = BBM(BB) * (1. + DLTS*(PB(BB)-BMB(BB)-PRB(BB)))

        END IF
C TEMPORARY FIX UP TO EXAMINE EFFECT OF SAV ON SEDIMENTS
66666   CONTINUE
C        NH4T2 = NH4T2TM1
C        PO4T2 = PO4T2TM1
C        NO3T2 = NO3T2TM1
C        HST2  = HST2TM1
C END TEMPORARY FIX UP

C TEMPORARY FIX UP WHEN SUSPENSION FEEDERS ARE RUN WITHOUT DIAGENESIS
C        BENDO(BB)  = - SODSF(BB)          ! suspension feeders
C        BENNH4(BB) = JNH4SF(BB)/1000.     ! suspension feeders
C        BENPO4(BB) = JPO4SF(BB)/1000.     ! suspension feeders
C        BENDOC(BB) = SF_LDOC(BB)/1000.
C END TEMPORARY FIX UP

******* Replace the t minus 1 concentrations

        NH41TM1S(BB)  = NH41
        NO31TM1S(BB)  = NO31
        HS1TM1S(BB)   = HS1
        PO41TM1S(BB)  = PO41
        BENSTR1S(BB)  = BENSTR
        NH4T2TM1S(BB) = NH4T2
        NO3T2TM1S(BB) = NO3T2
        HST2TM1S(BB)  = HST2
        PO4T2TM1S(BB) = PO4T2
        PON1TM1S(BB)  = PON1
        PON2TM1S(BB)  = PON2
        PON3TM1S(BB)  = PON3
        POC1TM1S(BB)  = POC1
        POC2TM1S(BB)  = POC2
        POC3TM1S(BB)  = POC3
        POP1TM1S(BB)  = POP1
        POP2TM1S(BB)  = POP2
        POP3TM1S(BB)  = POP3
        PIPTM1S(BB)   = SPIP
        BFORMAXS(BB)  = BFORMAX
        ISWBENS(BB)   = ISWBEN
        CH4T2TM1S(BB) = CH4T2               ! CH4
        CH41TM1S(BB)  = CH41                ! CH4
        SO4T2TM1S(BB) = SO4T2               ! CH4
        DIAGN(BB)     = XJN                 ! CFC

10080 CONTINUE

***** Assign concentrations to plot variables

      DO 10085 BB=1,NBB
        CPON(BB,1) = PON1TM1S(BB)
        CPON(BB,2) = PON2TM1S(BB)
        CPON(BB,3) = PON3TM1S(BB)
        CNH4(BB)   = NH4T2TM1S(BB)
        CNO3(BB)   = NO3T2TM1S(BB)
        CPOP(BB,1) = POP1TM1S(BB)
        CPOP(BB,2) = POP2TM1S(BB)
        CPOP(BB,3) = POP3TM1S(BB)
	CPIP(BB)   = PIPTM1S(BB)
        CPO4(BB)   = PO4T2TM1S(BB)
        CPOC(BB,1) = POC1TM1S(BB)
        CPOC(BB,2) = POC2TM1S(BB)
        CPOC(BB,3) = POC3TM1S(BB)
        CCH4(BB)   = CH4T2TM1S(BB)
        CSO4(BB)   = SO4T2TM1S(BB)
        CHS(BB)    = HST2TM1S(BB)                             !CFC
10085 CONTINUE

***** Take temperature integration step

      DO BB=1,NBB
        CTEMP(BB) = CTEMP(BB)+DLT*DIFFT/HSED(BB)/HSED(BB)
     .              *(T(BBN(BB))-CTEMP(BB))
      END DO
      
      RETURN

***** Compute and print out steady-state sediment concentrations

      ENTRY SED_INT

***** Compute time-average values

      DO 20000 BB=1,NBB
        AG3CFL(BB) = AG3CFL(BB)/TINTIM
        AG3NFL(BB) = AG3NFL(BB)/TINTIM
        AG3PFL(BB) = AG3PFL(BB)/TINTIM
        ASDTMP(BB) = ASDTMP(BB)/TINTIM
20000 CONTINUE

***** Compute G3 organic concentrations

      DO 20010 BB=1,NBB
        CPOC(BB,3) = AG3CFL(BB)/(KCDIAG(3)*DCTHTA(3)**(ASDTMP(BB)-20.)
     .               *HSED(BB)+VSED(BB))
        CPON(BB,3) = AG3NFL(BB)/(KNDIAG(3)*DNTHTA(3)**(ASDTMP(BB)-20.)
     .               *HSED(BB)+VSED(BB))
        CPOP(BB,3) = AG3PFL(BB)/(KPDIAG(3)*DPTHTA(3)**(ASDTMP(BB)-20.)
     .               *HSED(BB)+VSED(BB))
20010 CONTINUE

      RETURN
      END SUBROUTINE SED_CALC


********************************************************************************
**                          F U N C T I O N   S E D F                         **
********************************************************************************

      FUNCTION SEDF(SOD1) RESULT(SEDFOUT)
      ! <ezpp-noinst>
      USE WQM
      IMPLICIT NONE
! added next line 10/17/05
      SAVE    
      REAL SOD1, SEDFOUT
      REAL(8) AD(4,4), BX(4), G(2), H(2,2)
      REAL(8) DBLSO41, DBLSO42, RA0, RA1, RA2, R1, R2, DISC, SN1

***** Compute the NH4, NO3, and SOD fluxes

      S = SOD1/O20

***** Ammonia flux

      K0H1P = 0.
      K1H1P = 0.
      K2H2D = 0.
      K2H2P = 0.
      IF (KMNH4.NE.0.) THEN
        K0H1D = XAPPNH4**2/S*KMNH4*(O20/(KMNH4O2+O20))
        K1H1D = S
      ELSE
        K1H1D = XAPPNH4**2/S*(O20/(KMNH4O2+O20))+S
        K0H1D = 0.
      ENDIF
      J1   = S*NH40
      K3   = 0.
      J2   = XJN
      PIE1 = PIENH4
      PIE2 = PIENH4
      KMC1 = KMNH4
      CALL SEDTSFNL (NH41,NH42,NH4T1,NH4T2,NH41TM1,NH4T2TM1)
      JNH4 = S*(NH41-NH40)

***** Oxygen consumed by nitrification

      A1 = 0.0045714
      IF (KMNH4.NE.0.) THEN
        JO2NH4 = A1*K0H1D*NH41/(KMNH4+NH41TM1)
      ELSE
        JO2NH4 = A1*(K1H1D-S)*NH41
      ENDIF

***** Denitrification

      K0H1D = 0.
      K0H1P = 0.
      KMC1  = 0.
      K1H1D = XAPP1NO3 +S
      K1H1P = 0.
      K2H2D = XK2NO3
      K2H2P = 0.
      IF (KMNH4.NE.0.) THEN
        J1 = S*NO30+XAPPNH4**2/S*KMNH4*(O20/(KMNH4O2+O20))*NH41
     .       /(KMNH4+NH41TM1)
      ELSE
        J1 = S*NO30+XAPPNH4**2/S*(O20/(KMNH4O2+O20))*NH41
      ENDIF
      K3   = 0.
      J2   = 0.
      PIE1 = 0.
      PIE2 = 0.
      CALL SEDTSFNL(NO31,NO32,NO3T1,NO3T2,NO31TM1,NO3T2TM1)
      JNO3 = S*(NO31-NO30)

***** Sulfide/methane oxidation

      A2      = 0.00285714
      XJCNO31 = A2*XAPP1NO3*NO31
      XJCNO3  = A2*XK2NO3*NO32

***** Add the aerobic and first anaerobic layer to keep mass balance

      XJCNO3 = XJCNO31+XJCNO3

***** Convert carbon diagenesis flux to O2 units

      XJC1 = AMAX1(2.667E-3*XJC-XJCNO3,1.0e-10)
         
C**** **********************************************************
C**** New code for methane formation.  CH4 starts forming
C**** once all sulfate is used up.
C**** **********************************************************

C**** Sulfide and sulfate in O2 equivalents
C**** units: so4 in o2 equivalents
C     SO4 (mg so4/L)* 1 mmol SO4 /98 mg SO4 * 2 mmol O2/ 1 mmol SO4
C     * 32 mg O2 / mmol O2= 0.65306122

      SO40=SO40MG*0.65306122
      K0H1D=0.
      K0H1P=0.
      KMC1=0.0
      K1H1D=XAPPD1**2/S*(O20/KMHSO2) + S
      K1H1P=XAPPP1**2/S*(O20/KMHSO2)
      K2H2D=0.
      K2H2P=0.
      J1=0.
      K3=0.0
      J2=XJC1
      PIE1=PIE1S
      PIE2=PIE2S

C**** Set KL12 using H for SO4
      ITEMP = 10.*TEMPD+1
      DDSO4 = ZL12NOM(ITEMP)*H2
      HSO4  =SQRT(2.*DDSO4*SO40*H2/XJC1)

C**** No deeper than H2
      IF(HSO4.GT.H2) HSO4=H2
      KL12SO4=KL12*H2/HSO4

C**** Fractions and overall decay reaction velocity
      FD1=1./(1.+M1*PIE1)
      FP1=M1*PIE1/(1.+M1*PIE1)
      FD2=1./(1.+M2*PIE2)
      FP2=M2*PIE2/(1.+M2*PIE2)
      FP1SO4=FP1
      FP2SO4=FP2
      KHS_1=FP1*XAPPP1**2/S*(O20/KMHSO2)+FD1*XAPPD1**2/S*(O20/KMHSO2)

      BX(1) = DBLE(S)*DBLE(SO40)
      BX(2) = DBLE(H2)*DBLE(SO4T2TM1)/DBLE(DLTS)
      BX(3) = DBLE(HS0)*DBLE(S)
      BX(4) = DBLE(H2)*DBLE(HST2TM1)/DBLE(DLTS)

      AD(1,1) = -DBLE(S)-DBLE(KL12SO4)
      AD(1,2) = DBLE(KL12SO4)
      AD(1,3) = DBLE(KHS_1)
      AD(2,1) = DBLE(KL12SO4)
      AD(2,2) = -(DBLE(DLTS)*DBLE(KL12SO4)+DBLE(H2))/DBLE(DLTS)
      AD(3,3) = -DBLE(W2)-DBLE(FP1)*DBLE(W12)-DBLE(FD1)*DBLE(S)
     .          -DBLE(FD1)*DBLE(KL12SO4)-DBLE(KHS_1)
      AD(3,4) = DBLE(FP2)*DBLE(W12)+DBLE(FD2)*DBLE(KL12SO4)
      AD(4,3) = DBLE(W2)+DBLE(FP1)*DBLE(W12)+DBLE(FD1)*DBLE(KL12SO4)
      AD(4,4) = -(DBLE(DLTS)*DBLE(FP2)*DBLE(W12)
     .          +DBLE(DLTS)*DBLE(FD2)*DBLE(KL12SO4)+DBLE(DLTS)*DBLE(W2)
     .          +DBLE(H2)) /DBLE(DLTS)

      G(1) = ((BX(1)*AD(3,3) - AD(1,3)*BX(3))*AD(4,4) -
     .       BX(1)*AD(3,4)*AD(4,3) + AD(1,3)*AD(3,4)*BX(4) +
     .       AD(1,3)*BX(2)*AD(3,4))/(AD(1,3)*AD(3,4))

      G(2) = ((BX(1)*AD(3,3) - AD(1,3)*BX(3))*AD(4,4) -
     .       BX(1)*AD(3,4)*AD(4,3) + AD(1,3)*AD(3,4)*BX(4))
     .                             /(AD(1,3)*AD(3,4))

      H(1,1) = (AD(1,1)*AD(3,3)*AD(4,4)-AD(1,1)*AD(3,4)*AD(4,3)+AD(1,3)
     .     *AD(2,1)*AD(3,4))/(AD(1,3)*AD(3,4))
      H(1,2) = (AD(1,2)*AD(3,3)*AD(4,4)-AD(1,2)*AD(3,4)*AD(4,3)+AD(1,3)
     .     *AD(2,2)*AD(3,4))/(AD(1,3)*AD(3,4))
      H(2,1) = (AD(1,1)*AD(3,3)*AD(4,4)-AD(1,1)*AD(3,4)*
     .     AD(4,3))/(AD(1,3)*AD(3,4))
      H(2,2) = (AD(1,2)*AD(3,3)*AD(4,4)-AD(1,2)*AD(3,4)*
     .     AD(4,3))/(AD(1,3)*AD(3,4))

      RA0 = (H(1,1)*G(2)-G(1)*H(2,1))*DBLE(KMSO4)
      RA1 = - G(1)*H(2,1) + H(1,1)*G(2) +
     .      (H(1,1)*H(2,2)-H(1,2)*H(2,1))*DBLE(KMSO4) + H(1,1)*J2
      RA2 = H(1,1)*H(2,2)-H(1,2)*H(2,1)

      SN1 = 1.0D0                       !solution of a2*x^2+a1*x+a0
      IF (RA1.LE.0.0D0) SN1 = -1.0D0     !see Num Rec p178
      DISC = -(RA1+SN1*DSQRT(RA1**2-4.0D0*RA2*RA0))/2.0D0
      IF (DABS(DISC) /= 0.0D0) THEN
        R1 = DISC / RA2
        R2 = RA0 / DISC
      ELSE   ! vjp 11/17/2005 added logic for case (a2*a0) = 0
        IF (DABS(RA2) == 0.0D0) THEN   !  a2 = 0
           R1 = -RA0/RA1
           R2 = R1
        ELSE                           !  a0 = 0
           R1 = -RA1/RA2
           R2 = 0.0D0
        ENDIF
        print *, "R1 = ", R1," R2 = ", R2
      ENDIF

      DBLSO42 = R1
      IF (DBLSO42 .LT. 0.0D0) DBLSO42 = R2

      DBLSO41 = -(H(1,2)*DBLSO42+G(1))/H(1,1)
      HST1 = -(AD(1,2)*DBLSO42+AD(1,1)*DBLSO41+BX(1))/AD(1,3)
      HST2 = (AD(1,2)*AD(3,3)*DBLSO42+AD(1,1)*AD(3,3)*DBLSO41+BX(1)
     .       *AD(3,3)-AD(1,3)*BX(3))/(AD(1,3)*AD(3,4))
      HS1=FD1*HST1
      HS2=FD2*HST2
      HS2AV=FD2*HST2
      SO42=DBLSO42
      SO42AV=SO42
      SO4T2 = SO42
      SO41=DBLSO41
      XJ2=J2*KMSO4/(SO42+KMSO4)
      XJ2CH4=XJ2
      X1J2=J2*DBLSO42/(SO42+KMSO4)
      JHS=S*(HS1-HS0)
      CSODHS=(XAPPD1**2/S*FD1 + XAPPP1**2/S*FP1)*(O20/KMHSO2)*HST1

C**** Methane
      CH40 = 0.
      K0H1P=0.
      K1H1P=0.
      K2H2D=0.
      K2H2P=0.
      K1H1D=XAPPCH4**2/S*(O20/(KMCH4O2+O20))+S
      K0H1D=0.
      J1=S*CH40
      K3=0.0
      J2=XJ2
      PIE1=0.0
      PIE2=0.0
      KMC1=0.0

      CALL SEDSSFNL
     .(CH41,CH42,CH42AV,CH4T1,CH4T2,CH4T2AV,CH41TM1,CH4T2TM1,1)

      IF(CH42.GT.CH4SAT) THEN
         CH42=CH4SAT
         CH41 = (CH40*S**2+CH42*KL12*S)/(S**2+KL12*S+
     .           XAPPCH4**2*(O20/(KMCH4O2+O20)))
      ENDIF

C**** Calculate changes in CH4 and HS stored in the sediment
      DCH4T2 = (CH4T2 - CH4T2TM1)*H2/DLTS
      DHST2  = (HST2 - HST2TM1)*H2/DLTS

C**** Calculate CSOD
      CSODCH4 = XAPPCH4**2/S*(O20/(KMCH4O2+O20))*CH41
      CSOD    = CSODCH4+CSODHS

C**** Calculate Fluxes
      JCH4      = S*(CH41-CH40)
      JCH4AQ    = S*CH41
      FLUXHS    = S*FD1*HS1
      FLUXHSCH4 = JCH4AQ + FLUXHS

C**** If not flux or SOD or stored then it must escape as gas flux
      JCH4G = 0.
      IF (CH42.EQ.CH4SAT) THEN
        JCH4G = XJ2CH4 - DCH4T2 - CSODCH4 - JCH4AQ
      ENDIF

C**** Volumetric methane and total gas flux (L/m2-d)
      VJCH4G=22.4/64.*JCH4G
c     JGAS=JN2GAS+VJCH4G                   ! jn2gas not computed

***** SOD function

      SOD  = CSOD+JO2NH4
      SEDFOUT = SOD-SOD1

      RETURN
      END FUNCTION SEDF


********************************************************************************
**                        F U N C T I O N   Z B R E N T                       **
********************************************************************************

      FUNCTION ZBRENT(IERR) RESULT(ZBOUT)
      ! <ezpp-noinst>
      IMPLICIT NONE
      INTEGER,PARAMETER  :: IMAX=100
      REAL,PARAMETER     :: EPS=3.E-8, TOL=1.E-5
      INTEGER            :: IERR, I
      REAL               :: ZBOUT, SODMIN, SODMAX, TOL1
      REAL               :: A, B, C, D, E, P, Q, R, XM
      REAL               :: FA, FB, FC
C
      SODMIN = 1.E-4
      SODMAX = 100.
***** Initialize upper and lower limits for solution

      IERR = 0
      A    = SODMIN
      B    = SODMAX
      FA   = SEDF(A)
      FB   = SEDF(B)

***** Root must bracket ZBRENT

      IF (FB*FA.GT.0.) THEN
        IERR = 1
        RETURN
      ENDIF
      FC = FB
      DO 10000 I=1,IMAX
        IF (FB*FC.GT.0.) THEN
          C  = A
          FC = FA
          D  = B-A
          E  = D
        ENDIF
        IF (ABS(FC).LT.ABS(FB)) THEN
          A  = B
          B  = C
          C  = A
          FA = FB
          FB = FC
          FC = FA
        ENDIF
        TOL1 = 2.*EPS*ABS(B)+0.5*TOL
        XM   = 0.5*(C-B)
        IF (ABS(XM).LE.TOL1.OR.FB.EQ.0.) THEN
          ZBOUT  = B
          RETURN
        ENDIF
        IF (ABS(E).GE.TOL1.AND.ABS(FA).GT.ABS(FB)) THEN
          S = FB/FA
          IF (A.EQ.C) THEN
            P = 2.*XM*S
            Q = 1.-S
          ELSE
            Q = FA/FC
            R = FB/FC
            P = S*(2.*XM*Q*(Q-R)-(B-A)*(R-1.))
            Q = (Q-1.)*(R-1.)*(S-1.)
          ENDIF
          IF (P.GT.0.) Q = -Q
          P = ABS(P)
          IF (2.*P.LT.MIN(3.*XM*Q-ABS(TOL1*Q),ABS(E*Q))) THEN
            E = D
            D = P/Q
          ELSE
            D = XM
            E = D
          ENDIF
        ELSE
          D = XM
          E = D
        ENDIF
        A  = B
        FA = FB
        IF (ABS(D).GT.TOL1) THEN
          B = B+D
        ELSE
          B = B+SIGN(TOL1,XM)
        ENDIF
        FB = SEDF(B)
10000 CONTINUE
      IERR   = 2
      ZBOUT  = B
      RETURN
      END FUNCTION ZBRENT



********************************************************************************
**                    S U B R O U T I N E   S E D T S F N L                   **
********************************************************************************

      SUBROUTINE SEDTSFNL(C1S,C2S,CT1S,CT2S,C1TM1S,CT2TM1S)
      ! <ezpp-noinst>
      USE WQM;  USE FILE_INFO
      IMPLICIT NONE
      REAL :: C1S,C2S,CT1S,CT2S,C1TM1S,CT2TM1S
      REAL :: A11, A12, A21, A22, B_1, B_2
      REAL :: F12, F21, XK0, XK1, XK2, DELTA

***** Initialize constants
     
      FD1 = 1./(1.+M1*PIE1)
      FP1 = M1*PIE1/(1.+M1*PIE1)
      FD2 = 1./(1.+M2*PIE2)
      FP2 = M2*PIE2/(1.+M2*PIE2)
      F12 = W12*FP1+KL12*FD1
      F21 = W12*FP2+KL12*FD2

***** Evaluate the MM term at time level t-1

      IF (KMC1.NE.0.) THEN
        XK0 = (K0H1D*FD1+K0H1P*FP1)/(KMC1+C1TM1S)
      ELSE
        XK0 = 0.
      ENDIF
      XK1 = XK0+K1H1D*FD1+K1H1P*FP1
      XK2 = K2H2D*FD2+K2H2P*FP2
      A11 = -F12-XK1-W2
      A21 = F12+W2
      A12 = F21
      B_1 = -J1
      A22 = -F21-XK2-W2-K3-H2/DLTS
      B_2 = -J2-H2/DLTS*CT2TM1S

***** Solve the 2x2 set of linear equations

      DELTA = A11*A22-A12*A21
      IF (DELTA.EQ.0.) THEN
        PRINT *,'Twod is singular: A11,A12,A21,A22'
        write(*,*) f12, xk1, w2
        write(*,*) w12, fp1, kl12, fd1
        write(*,*) xk0, k1h1d, fd1, k1h1p, fp1
        
        PRINT *,A11,A12,A21,A22
        STOP
      ENDIF

***** Assign results

      CT1S = (B_1*A22-B_2*A12)/DELTA
      CT2S = (B_2*A11-B_1*A21)/DELTA
      C1S  = FD1*CT1S
      C2S  = FD2*CT2S
      RETURN
      END SUBROUTINE SEDTSFNL


********************************************************************************
**                    S U B R O U T I N E   S E D S S F N L                   **
********************************************************************************

      SUBROUTINE SEDSSFNL(C1,C2,C2AV,CT1,CT2,CT2AV,C1TM1,CT2TM1,ITYPE)
      ! <ezpp-noinst>
      IMPLICIT NONE
      INTEGER ITYPE
      REAL C1,C2,C2AV,CT1,CT2,CT2AV,C1TM1,CT2TM1

C     This subroutine translates between SEDTSFNL and SEDF
C     This is called by some sections of the new code

      CALL SEDTSFNL (C1,C2,CT1,CT2,C1TM1,CT2TM1)

      C2AV  = C2
      CT2AV = CT2

      RETURN
      END SUBROUTINE SEDSSFNL


      END MODULE SED
